using System;
using System.Collections.Generic;

namespace HIPS.Web.BusinessLogic.Identification
{
    public static class MedicareValidator
    {
        private const int CardLength = 10;
        private const int MinFirstDigit = 2;
        private const int MaxFirstDigit = 6;

        public static List<string> ValidateMedicareNumber(string medicareNumber, string medicareIrn)
        {
            List<string> errors = new List<string>();

            if (!IsValidMedicareNumber(medicareNumber))
            {
                errors.Add("Medicare Number must be a valid Medicare Card Number.");
            }

            if (!String.IsNullOrWhiteSpace(medicareIrn))
            {
                if (!IsValidIrn(medicareIrn))
                {
                    errors.Add("Medicare IRN must be a single digit between 1 and 9.");
                }
            }
            else
            {
                errors.Add("Medicare IRN must be provided for Medicare Numbers.");
            }

            return errors;
        }

        public static bool IsValidIrn(string irn)
        {
            if (String.IsNullOrWhiteSpace(irn))
            {
                return false;
            }

            int medicareIrnNumeric;
            if (!int.TryParse(irn, out medicareIrnNumeric))
            {
                return false;
            }

            return medicareIrnNumeric >= 1 && medicareIrnNumeric <= 9;
        }

        /// <summary>
        ///     Validates the specified Medicare card number is in a valid format.
        /// </summary>
        /// <param name="cardNumber">The card number.</param>
        /// <returns>
        ///     If true is returned then the Medicare Number has passed the following format rules:
        ///     It has 10 digits
        ///     The first digit is 2,3,4,5 or 6.
        ///     Digit 9 is a valid check digit:
        ///     a) Calculate the sum of: ((digit 1) + (digit 2 * 3) + (digit 3 * 7) + (digit 4 * 9) + (digit 5) + (digit 6 * 3) + (digit 7 * 7) + (digit 8 * 9))
        ///     b) Remainder of Mod 10 of this calculated number is the check digit.
        ///     If false is returned then this will be treated as an invalid Medicare Number.
        /// </returns>
        public static bool IsValidMedicareNumber(string cardNumber)
        {
            // Card number not blank
            if (String.IsNullOrWhiteSpace(cardNumber))
            {
                return false;
            }

            // Card number length is 10
            string validCardNumber = cardNumber.Trim();
            if (validCardNumber.Length != CardLength)
            {
                return false;
            }

            // Check if each character is an ASCII character 
            // representing a number between 0 and 9 inclusive.
            int[] digit = new int[CardLength + 1];
            for (int i = 1; i <= CardLength; i++)
            {
                digit[i] = validCardNumber[i - 1] - '0';

                if (digit[i] < 0 || digit[i] > 9)
                {
                    return false;
                }
            }

            // First digit is between 2 and 6 inclusive.
            if (digit[1] < MinFirstDigit || digit[1] > MaxFirstDigit)
            {
                return false;
            }

            // Checksum
            int checkSum = (digit[1] + (digit[2] * 3) + (digit[3] * 7) + (digit[4] * 9) + digit[5] + (digit[6] * 3) + (digit[7] * 7) + (digit[8] * 9));
            int remainder = checkSum % 10;
            bool result = (remainder == digit[9]);

            return result;
        }
    }
}